/***************************************************************
 *                 Mathematical Object Library                 *
 *                 class gaussian white noise                  *
 *                    simula.plus@cemes.fr                     *
 *	             GNU/linux version 3.2.0	               *
 *            software under General Public License            *
 ***************************************************************
 * copyright © 2012 COLLARD Christophe
 * copyright © 2012 Centre National de la Recherche Scientifique
 * copyright © 2012 Centre d'Elaboration de Matériaux et d'Etudes Structurales (CEMES - CNRS)
 ***************************************************************/

/*! \namespace mol
    \brief Mathematical Object Libraries
*/

/*! \class mol::gaussian_white_noise
    \brief gaussian white noise library \n

    \htmlonly 
    <FONT color="#838383">

    gaussian_white_noise belongs to Mathematical Object Libraries (MOL++) </br>
    MOL++ is part of Simula+ <br><br>

    Simula+ is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version. <br><br>

    Simula+ is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details. <br><br>

    You should have received a copy of the GNU General Public License
    along with Simula+; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

    </FONT>
    \endhtmlonly

    \authors copyright \htmlonly &#169; \endhtmlonly 2012 Christophe COLLARD \n
             copyright \htmlonly &#169; 2012 Centre National de la Recherche Scientifique \endhtmlonly \n
	     copyright \htmlonly &#169; 2012 Centre d'Elaboration de Mat&#233;riaux et d'Etudes Structurales (CEMES - CNRS) \endhtmlonly \n
    \version 3.2.0
    \date 2012
    \bug none
    \warning none
*/

#ifndef __cplusplus
#error Must use C++ for the type vector
#endif

#ifndef __gaussian_white_noise_hpp
#define __gaussian_white_noise_hpp


#ifndef __iostream
#include <iostream>
#endif

#ifndef __cstdlib
#include <cstdlib>
#endif

#ifndef __ctime
#include <ctime>
#endif

#ifndef __matrix_hpp
#include "matrix.hpp"
#endif

using namespace std;

namespace mol
{


//===========================================
template <class T> class gaussian_white_noise
//===========================================
{
  private:
    int nb_elements;
    int current_element;
    matrix<T> random_numbers;

  public:
    gaussian_white_noise (int);

    T get_random_number ();
    matrix<T> get_all_numbers () {return random_numbers;}
};


//=====Private methods for gaussian white noise==============================



//=====Public methods for gaussian white noise===============================


//-----------------------------------------------------------------------
template <class T> gaussian_white_noise<T>::gaussian_white_noise (int nb)
//-----------------------------------------------------------------------
{
  current_element = 0;
  nb_elements = nb;
  nb = 0.5 * (nb+1);
  random_numbers = matrix<T> (2 * nb, 1);
  srand ( time (NULL) );
  for (int i=1; i<=nb; i++)
    { long double S = 1;
      long double U1, U2, V1, V2;
      for (; !S || S>=1;)
	{ U1 = rand () / (long double) RAND_MAX; /* U1=[0,1] */
	  U2 = rand () / (long double) RAND_MAX; /* U2=[0,1] */
	  V1 = 2 * U1 - 1; /* V1=[-1,1] */
	  V2 = 2 * U2 - 1; /* V2=[-1,1] */
	  S = V1 * V1 + V2 * V2;
	}

      random_numbers (i,1) = sqrt (- 2 * log (S) / S) * V1;
      random_numbers (nb+i, 1) = sqrt (- 2 * log (S) / S) * V2;
    }
}

//-----------------------------------------------------------------
template <class T>  T gaussian_white_noise<T>::get_random_number ()
//-----------------------------------------------------------------
{
  assert (++current_element < nb_elements); // the number of created element with the current seed is too small
  return random_numbers (current_element, 1);
}


}


#endif
